$dscModuleRoot = Split-Path -Path (Split-Path -Path $PSScriptRoot -Parent) -Parent

# Import the Common Modules
Import-Module -Name (Join-Path -Path $dscModuleRoot -ChildPath 'Modules\RemoteDesktopServicesDsc.Common') -ErrorAction Stop
Import-Module -Name (Join-Path -Path $dscModuleRoot -ChildPath 'Modules\DscResource.Common') -ErrorAction Stop

$env:PSModulePath = [Environment]::GetEnvironmentVariable("PSModulePath", "Machine") + ";" + $env:PSModulePath

function Sync-RDSModule {
    $manifest = Join-Path $env:SystemRoot "system32\WindowsPowerShell\v1.0\Modules\RemoteDesktop\RemoteDesktop.psd1"
    # FIX: Separated Test-Path and Get-Module to avoid the 'and' parameter error
    if (Test-Path -Path $manifest) {
        if (-not (Get-Module -Name RemoteDesktop)) {
            Import-Module -Name $manifest -ErrorAction SilentlyContinue
        }
    }
}

#######################################################################
# The Get-TargetResource cmdlet.
#######################################################################
function Get-TargetResource
{
    [CmdletBinding()]
    [OutputType([System.Collections.Hashtable])]
    param
    (
        [Parameter(Mandatory = $true)]
        [System.String] $ConnectionBroker,
        [Parameter(Mandatory = $true)]
        [System.String] $LicenseMode,
        [Parameter()]
        [System.String[]] $LicenseServer
    )

    Sync-RDSModule
    $config = RemoteDesktop\Get-RDLicenseConfiguration -ConnectionBroker $ConnectionBroker -ErrorAction SilentlyContinue

    if ($config)
    {
        # SERVER 2025 FIX: Map the Enum mode to the DSC string format
        $currentMode = switch ($config.Mode) {
            2 { "PerDevice" }
            4 { "PerUser" }
            0 { "NotConfigured" }
            Default { $config.Mode.ToString().Replace(" ", "") }
        }

        return @{
            ConnectionBroker = $ConnectionBroker
            LicenseMode      = $currentMode
            LicenseServer    = [System.String[]]$config.LicenseServer
        }
    }
    return @{
        ConnectionBroker = $ConnectionBroker
        LicenseMode      = "NotConfigured"
        LicenseServer    = @()
    }
}

########################################################################
# The Set-TargetResource cmdlet.
########################################################################
function Set-TargetResource
{
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [System.String] $ConnectionBroker,

        [Parameter(Mandatory = $true)]
        [System.String] $LicenseMode,

        [Parameter()]
        [System.String[]] $LicenseServer
    )

    Sync-RDSModule
    $setLicenseConfigParams = @{
        ConnectionBroker = $ConnectionBroker
        Mode             = $LicenseMode
    }
    if ($LicenseServer) { $setLicenseConfigParams.LicenseServer = $LicenseServer }

    RemoteDesktop\Set-RDLicenseConfiguration @setLicenseConfigParams -Force
}

#######################################################################
# The Test-TargetResource cmdlet.
#######################################################################
function Test-TargetResource
{
    [CmdletBinding()]
    [OutputType([System.Boolean])]
    param
    (
        [Parameter(Mandatory = $true)]
        [System.String] $ConnectionBroker,
        [Parameter(Mandatory = $true)]
        [System.String] $LicenseMode,
        [Parameter()]
        [System.String[]] $LicenseServer
    )

    $current = Get-TargetResource -ConnectionBroker $ConnectionBroker -LicenseMode $LicenseMode
    
    # Logic Check: If GUI shows empty and we want 'PerUser', this will trigger 'Set'
    if ($current.LicenseMode -ne $LicenseMode) {
        Write-Verbose "Mode Mismatch: Current is $($current.LicenseMode), Desired is $LicenseMode"
        return $false
    }

    # Compare Servers (Sort both to ensure order doesn't trip it up)
    $currentServers = $current.LicenseServer | Sort-Object
    $desiredServers = $LicenseServer | Sort-Object
    
    if (Compare-Object -ReferenceObject $currentServers -DifferenceObject $desiredServers) {
        Write-Verbose "Server List Mismatch"
        return $false
    }

    return $true
}
Export-ModuleMember -Function *-TargetResource