$modulePath = Join-Path -Path (Split-Path -Path (Split-Path -Path $PSScriptRoot -Parent) -Parent) -ChildPath 'Modules'

# Import the Common Modules
Import-Module -Name (Join-Path -Path $modulePath -ChildPath 'RemoteDesktopServicesDsc.Common')
Import-Module -Name (Join-Path -Path $modulePath -ChildPath 'DscResource.Common')

# SERVER 2025 FIX: Force path refresh for non-interactive sessions
$env:PSModulePath = [Environment]::GetEnvironmentVariable("PSModulePath", "Machine") + ";" + $env:PSModulePath

if (-not (Test-RemoteDesktopServicesDscOsRequirement))
{
    throw 'The minimum OS requirement was not met.'
}

# Helper to bypass discovery cache issues in Server 2025
function Sync-RDSModule {
    $manifest = Join-Path $env:SystemRoot "system32\WindowsPowerShell\v1.0\Modules\RemoteDesktop\RemoteDesktop.psd1"
    if (Test-Path $manifest) {
        Import-Module -Name $manifest -ErrorAction SilentlyContinue
    }
}

#######################################################################
# The Get-TargetResource cmdlet.
#######################################################################
function Get-TargetResource
{
    [CmdletBinding()]
    [OutputType([System.Collections.Hashtable])]
    param
    (
        [Parameter(Mandatory = $true)]
        [ValidateLength(1, 256)]
        [System.String] $CollectionName,

        [Parameter(Mandatory = $true)]
        [System.String[]] $SessionHost,

        [Parameter()]
        [System.String] $CollectionDescription,

        [Parameter(Mandatory = $true)]
        [System.String] $ConnectionBroker,

        [Parameter()]
        [System.Boolean] $Force
    )

    Write-Verbose -Message 'Getting information about RDSH collection.'

    # SERVER 2025 FIX: Use direct manifest loading
    Sync-RDSModule

    $params = @{
        ConnectionBroker = $ConnectionBroker
        CollectionName   = $CollectionName
        ErrorAction      = 'SilentlyContinue'
    }

    # SERVER 2025 FIX: Fully Qualified Name
    $Collection = RemoteDesktop\Get-RDSessionCollection @params | `
                  Where-Object CollectionName -EQ $CollectionName

    if ($Collection.Count -eq 0)
    {
        return @{
            CollectionName        = $null
            ConnectionBroker      = $null
            CollectionDescription = $null
            SessionHost           = [System.String[]] $SessionHost
            Force                 = $Force
        }
    }

    if ($Collection.Count -gt 1)
    {
        New-InvalidResultException -Message 'Non-singular RDSessionCollection in result set'
    }

    # SERVER 2025 FIX: Fully Qualified Name for Get-RDSessionHost
    $hosts = RemoteDesktop\Get-RDSessionHost -CollectionName $CollectionName -ConnectionBroker $ConnectionBroker -ErrorAction SilentlyContinue

    return @{
        CollectionName        = $Collection.CollectionName
        ConnectionBroker      = $ConnectionBroker
        CollectionDescription = $Collection.CollectionDescription
        SessionHost           = [System.String[]] ($hosts.SessionHost)
        Force                 = $Force
    }
}

########################################################################
# The Set-TargetResource cmdlet.
########################################################################
function Set-TargetResource
{
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [ValidateLength(1, 256)]
        [System.String] $CollectionName,

        [Parameter(Mandatory = $true)]
        [System.String[]] $SessionHost,

        [Parameter()]
        [System.String] $CollectionDescription,

        [Parameter(Mandatory = $true)]
        [System.String] $ConnectionBroker,

        [Parameter()]
        [System.Boolean] $Force
    )

    Sync-RDSModule

    $currentStatus = Get-TargetResource @PSBoundParameters

    if ($null -ne $currentStatus.CollectionName -and $Force)
    {
        $missing, $surplus = @()
        Write-Verbose -Message "Session collection $CollectionName already exists. Updating Session Hosts."
        if ($null -ne $currentStatus.SessionHost)
        {
            $compare = Compare-Object -ReferenceObject $SessionHost -DifferenceObject $currentStatus.SessionHost -PassThru
            $surplus, $missing = $compare.Where({ $_.SideIndicator -eq '=>' }, 'Split')
        }
        else
        {
            $missing = $SessionHost
        }

        foreach ($server in $missing)
        {
            # SERVER 2025 FIX: Fully Qualified Name
            RemoteDesktop\Add-RDSessionHost -CollectionName $CollectionName -ConnectionBroker $ConnectionBroker -SessionHost $server
        }

        foreach ($server in $surplus)
        {
            # SERVER 2025 FIX: Fully Qualified Name
            RemoteDesktop\Remove-RDSessionHost -ConnectionBroker $ConnectionBroker -SessionHost $server -Force
        }

        return
    }

    $newCollectionParams = @{
        CollectionName        = $CollectionName
        CollectionDescription = $CollectionDescription
        ConnectionBroker      = $ConnectionBroker
        SessionHost           = $SessionHost
    }

    $exception = $null

    try
    {
        Write-Verbose -Message 'Creating a new RDSH collection.'
        # SERVER 2025 FIX: Fully Qualified Name
        RemoteDesktop\New-RDSessionCollection @newCollectionParams -ErrorAction Stop
    }
    catch
    {
        $exception = $_.Exception
    }

    if (-not (Test-TargetResource @PSBoundParameters))
    {
        $exceptionString = ("'Test-TargetResource' returns false after call to 'New-RDSessionCollection'; CollectionName: {0}; ConnectionBroker {1}." -f $CollectionName, $ConnectionBroker)
        Write-Verbose -Message $exceptionString

        if ($exception)
        {
            $exception = [System.Management.Automation.RuntimeException]::new($exceptionString, $exception)
        }
        else
        {
            $exception = [System.Management.Automation.RuntimeException]::new($exceptionString)
        }

        $PSCmdlet.ThrowTerminatingError([System.Management.Automation.ErrorRecord]::new($exception, 'Failure to coerce resource into the desired state', [System.Management.Automation.ErrorCategory]::InvalidResult, $CollectionName))
    }
}

#######################################################################
# The Test-TargetResource cmdlet.
#######################################################################
function Test-TargetResource
{
    [CmdletBinding()]
    [OutputType([System.Boolean])]
    param
    (
        [Parameter(Mandatory = $true)]
        [ValidateLength(1, 256)]
        [System.String] $CollectionName,

        [Parameter(Mandatory = $true)]
        [System.String[]] $SessionHost,

        [Parameter()]
        [System.String] $CollectionDescription,

        [Parameter(Mandatory = $true)]
        [System.String] $ConnectionBroker,

        [Parameter()]
        [System.Boolean] $Force
    )

    $testDscParameterStateSplat = @{
        CurrentValues       = Get-TargetResource @PSBoundParameters
        DesiredValues       = $PSBoundParameters
        TurnOffTypeChecking = $false
        SortArrayValues     = $true
        Verbose             = $VerbosePreference
    }

    return Test-DscParameterState @testDscParameterStateSplat
}

Export-ModuleMember -Function *-TargetResource