$modulePath = Join-Path -Path (Split-Path -Path (Split-Path -Path $PSScriptRoot -Parent) -Parent) -ChildPath 'Modules'

# Import the Common Modules
Import-Module -Name (Join-Path -Path $modulePath -ChildPath 'RemoteDesktopServicesDsc.Common')
Import-Module -Name (Join-Path -Path $modulePath -ChildPath 'DscResource.Common')

# SERVER 2025 FIX: Ensure environment path is current for the RunAs session
$env:PSModulePath = [Environment]::GetEnvironmentVariable("PSModulePath", "Machine") + ";" + $env:PSModulePath

if (-not (Test-RemoteDesktopServicesDscOsRequirement))
{
    throw 'The minimum OS requirement was not met.'
}

# Helper to bypass discovery cache issues
function Sync-RDSModule {
    $manifest = Join-Path $env:SystemRoot "system32\WindowsPowerShell\v1.0\Modules\RemoteDesktop\RemoteDesktop.psd1"
    if (Test-Path $manifest) {
        Import-Module -Name $manifest -ErrorAction SilentlyContinue
    }
}

#######################################################################
# The Get-TargetResource cmdlet.
#######################################################################
function Get-TargetResource
{
    [CmdletBinding()]
    [OutputType([System.Collections.Hashtable])]
    param
    (
        [Parameter(Mandatory = $true)]
        [ValidateLength(1, 256)]
        [System.String] $CollectionName,
        [System.UInt32] $ActiveSessionLimitMin,
        [System.Boolean] $AuthenticateUsingNLA,
        [System.Boolean] $AutomaticReconnectionEnabled,
        [System.String] $BrokenConnectionAction,
        [System.String] $ClientDeviceRedirectionOptions,
        [System.Boolean] $ClientPrinterAsDefault,
        [System.Boolean] $ClientPrinterRedirected,
        [System.String] $CollectionDescription,
        [System.String] $ConnectionBroker,
        [System.String] $CustomRdpProperty,
        [System.UInt32] $DisconnectedSessionLimitMin,
        [System.String] $EncryptionLevel,
        [System.UInt32] $IdleSessionLimitMin,
        [System.UInt32] $MaxRedirectedMonitors,
        [System.Boolean] $RDEasyPrintDriverEnabled,
        [System.String] $SecurityLayer,
        [System.Boolean] $TemporaryFoldersDeletedOnExit,
        [System.String[]] $UserGroup,
        [System.String] $DiskPath,
        [System.Boolean] $EnableUserProfileDisk,
        [System.UInt32] $MaxUserProfileDiskSizeGB,
        [System.String[]] $IncludeFolderPath,
        [System.String[]] $ExcludeFolderPath,
        [System.String[]] $IncludeFilePath,
        [System.String[]] $ExcludeFilePath
    )

    # SERVER 2025 FIX: Load manifest
    Sync-RDSModule

    Write-Verbose "Getting currently configured RDSH Collection properties for collection $CollectionName"

    # SERVER 2025 FIX: Prefix all Get calls
    $collectionGeneral = RemoteDesktop\Get-RDSessionCollectionConfiguration -CollectionName $CollectionName
    $collectionClient = RemoteDesktop\Get-RDSessionCollectionConfiguration -CollectionName $CollectionName -Client
    $collectionConnection = RemoteDesktop\Get-RDSessionCollectionConfiguration -CollectionName $CollectionName -Connection
    $collectionSecurity = RemoteDesktop\Get-RDSessionCollectionConfiguration -CollectionName $CollectionName -Security
    $collectionUserGroup = RemoteDesktop\Get-RDSessionCollectionConfiguration -CollectionName $CollectionName -UserGroup

    $result = @{
        CollectionName                 = $collectionGeneral.CollectionName
        CollectionDescription          = $collectionGeneral.CollectionDescription
        CustomRdpProperty              = ([System.String]$collectionGeneral.CustomRdpProperty).Trim()
        ClientDeviceRedirectionOptions = $collectionClient.ClientDeviceRedirectionOptions
        ClientPrinterAsDefault         = $collectionClient.ClientPrinterAsDefault
        ClientPrinterRedirected        = $collectionClient.ClientPrinterRedirected
        MaxRedirectedMonitors          = $collectionClient.MaxRedirectedMonitors
        RDEasyPrintDriverEnabled       = $collectionClient.RDEasyPrintDriverEnabled
        ActiveSessionLimitMin          = $collectionConnection.ActiveSessionLimitMin
        AutomaticReconnectionEnabled   = $collectionConnection.AutomaticReconnectionEnabled
        BrokenConnectionAction         = $collectionConnection.BrokenConnectionAction
        DisconnectedSessionLimitMin    = $collectionConnection.DisconnectedSessionLimitMin
        IdleSessionLimitMin            = $collectionConnection.IdleSessionLimitMin
        TemporaryFoldersDeletedOnExit  = $collectionConnection.TemporaryFoldersDeletedOnExit
        AuthenticateUsingNLA           = $collectionSecurity.AuthenticateUsingNLA
        EncryptionLevel                = $collectionSecurity.EncryptionLevel
        SecurityLayer                  = $collectionSecurity.SecurityLayer
        UserGroup                      = $collectionUserGroup.UserGroup
    }

    # OS Major 10 covers Server 2016 through 2025
    if ((Get-RemoteDesktopServicesDscOsVersion).Major -ge 10)
    {
        Write-Verbose 'Running on W2016+, get UserProfileDisk configuration'
        $collectionUserProfileDisk = RemoteDesktop\Get-RDSessionCollectionConfiguration -CollectionName $CollectionName -UserProfileDisk

        $null = $result.Add('DiskPath', $collectionUserProfileDisk.DiskPath)
        $null = $result.Add('EnableUserProfileDisk', $collectionUserProfileDisk.EnableUserProfileDisk)
        $null = $result.Add('MaxUserProfileDiskSizeGB', $collectionUserProfileDisk.MaxUserProfileDiskSizeGB)
        $null = $result.Add('IncludeFolderPath', $collectionUserProfileDisk.IncludeFolderPath)
        $null = $result.Add('ExcludeFolderPath', $collectionUserProfileDisk.ExcludeFolderPath)
        $null = $result.Add('IncludeFilePath', $collectionUserProfileDisk.IncludeFilePath)
        $null = $result.Add('ExcludeFilePath', $collectionUserProfileDisk.ExcludeFilePath)
    }

    $result
}

########################################################################
# The Set-TargetResource cmdlet.
########################################################################
function Set-TargetResource
{
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [ValidateLength(1, 256)]
        [System.String] $CollectionName,
        [System.UInt32] $ActiveSessionLimitMin,
        [System.Boolean] $AuthenticateUsingNLA,
        [System.Boolean] $AutomaticReconnectionEnabled,
        [System.String] $BrokenConnectionAction,
        [System.String] $ClientDeviceRedirectionOptions,
        [System.Boolean] $ClientPrinterAsDefault,
        [System.Boolean] $ClientPrinterRedirected,
        [System.String] $CollectionDescription,
        [System.String] $ConnectionBroker,
        [System.String] $CustomRdpProperty,
        [System.UInt32] $DisconnectedSessionLimitMin,
        [System.String] $EncryptionLevel,
        [System.UInt32] $IdleSessionLimitMin,
        [System.UInt32] $MaxRedirectedMonitors,
        [System.Boolean] $RDEasyPrintDriverEnabled,
        [System.String] $SecurityLayer,
        [System.Boolean] $TemporaryFoldersDeletedOnExit,
        [System.String[]] $UserGroup,
        [System.String] $DiskPath,
        [System.Boolean] $EnableUserProfileDisk,
        [System.UInt32] $MaxUserProfileDiskSizeGB,
        [System.String[]] $IncludeFolderPath,
        [System.String[]] $ExcludeFolderPath,
        [System.String[]] $IncludeFilePath,
        [System.String[]] $ExcludeFilePath
    )

    Sync-RDSModule

    Write-Verbose 'Setting DSC collection properties'

    try {
        # SERVER 2025 FIX: Prefix lookup
        $null = RemoteDesktop\Get-RDSessionCollection -CollectionName $CollectionName -ErrorAction Stop
    }
    catch {
        throw "Failed to lookup RD Session Collection $CollectionName. Error: $_"
    }

    # Clean up parameters for the initial splat
    $updParams = @('DiskPath','EnableUserProfileDisk','ExcludeFilePath','ExcludeFolderPath','IncludeFilePath','IncludeFolderPath','MaxUserProfileDiskSizeGB')
    foreach ($p in $updParams) { $null = $PSBoundParameters.Remove($p) }

    if ((Get-RemoteDesktopServicesDscOsVersion).Major -ge 10)
    {
        Write-Verbose 'Running on W2016 or higher, set configuration'

        # SERVER 2025 FIX: Prefix Set call
        RemoteDesktop\Set-RDSessionCollectionConfiguration @PSBoundParameters

        if ($EnableUserProfileDisk)
        {
            if ($DiskPath -and -not(Test-Path -Path $DiskPath -ErrorAction SilentlyContinue))
            {
                New-ArgumentException -ArgumentName 'DiskPath' -Message ('To enable UserProfileDisk we need a valid DiskPath. Path {0} not found' -f $DiskPath)
            }

            $enableUserProfileDiskSplat = @{
                CollectionName           = $CollectionName
                DiskPath                 = $DiskPath
                EnableUserProfileDisk    = $EnableUserProfileDisk
                ExcludeFilePath          = $ExcludeFilePath
                ExcludeFolderPath        = $ExcludeFolderPath
                IncludeFilePath          = $IncludeFilePath
                IncludeFolderPath        = $IncludeFolderPath
                MaxUserProfileDiskSizeGB = $MaxUserProfileDiskSizeGB
            }

            # SERVER 2025 FIX: Prefix the UPD configuration call
            $null = RemoteDesktop\Set-RDSessionCollectionConfiguration @enableUserProfileDiskSplat -ErrorAction SilentlyContinue -ErrorVariable setRDSessionCollectionErrors 2>&1

            # Trapping the CommandNotFoundException that occurs inside Microsoft's own internal workflow calls
            foreach ($err in $setRDSessionCollectionErrors)
            {
                if ($err.FullyQualifiedErrorId -ne 'CommandNotFoundException')
                {
                    Write-Error "Set-RDSessionCollectionConfiguration error: $err"
                }
            }
        }
        else
        {
            # SERVER 2025 FIX: Prefix Disable call
            RemoteDesktop\Set-RDSessionCollectionConfiguration -CollectionName $CollectionName -DisableUserProfileDisk
        }
    }
    else
    {
        RemoteDesktop\Set-RDSessionCollectionConfiguration @PSBoundParameters
    }
}

#######################################################################
# The Test-TargetResource cmdlet.
#######################################################################
function Test-TargetResource
{
    [CmdletBinding()]
    [OutputType([System.Boolean])]
    param
    (
        [Parameter(Mandatory = $true)]
        [ValidateLength(1, 256)]
        [System.String] $CollectionName,

        [Parameter()]
        [System.UInt32] $ActiveSessionLimitMin,

        [Parameter()]
        [System.Boolean] $AuthenticateUsingNLA,

        [Parameter()]
        [System.Boolean] $AutomaticReconnectionEnabled,

        [Parameter()]
        [System.String] $BrokenConnectionAction,

        [Parameter()]
        [System.String] $ClientDeviceRedirectionOptions,

        [Parameter()]
        [System.Boolean] $ClientPrinterAsDefault,

        [Parameter()]
        [System.Boolean] $ClientPrinterRedirected,

        [Parameter()]
        [System.String] $CollectionDescription,

        [Parameter()]
        [System.String] $ConnectionBroker,

        [Parameter()]
        [System.String] $CustomRdpProperty,

        [Parameter()]
        [System.UInt32] $DisconnectedSessionLimitMin,

        [Parameter()]
        [System.String] $EncryptionLevel,

        [Parameter()]
        [System.UInt32] $IdleSessionLimitMin,

        [Parameter()]
        [System.UInt32] $MaxRedirectedMonitors,

        [Parameter()]
        [System.Boolean] $RDEasyPrintDriverEnabled,

        [Parameter()]
        [System.String] $SecurityLayer,

        [Parameter()]
        [System.Boolean] $TemporaryFoldersDeletedOnExit,

        [Parameter()]
        [System.String[]] $UserGroup,

        [Parameter()]
        [System.String] $DiskPath,

        [Parameter()]
        [System.Boolean] $EnableUserProfileDisk,

        [Parameter()]
        [System.String[]] $ExcludeFilePath,

        [Parameter()]
        [System.String[]] $ExcludeFolderPath,

        [Parameter()]
        [System.String[]] $IncludeFilePath,

        [Parameter()]
        [System.String[]] $IncludeFolderPath,

        [Parameter()]
        [System.UInt32] $MaxUserProfileDiskSizeGB
    )

    Write-Verbose 'Testing DSC collection properties'

    # We must remove these internal DSC parameters so they don't 
    # interfere with the property comparison logic.
    $null = $PSBoundParameters.Remove('Verbose')
    $null = $PSBoundParameters.Remove('Debug')

    # ConnectionBroker is a 'Write' property in the MOF but it's used 
    # as an identifier to fetch the current state.
    $cb = $ConnectionBroker
    $null = $PSBoundParameters.Remove('ConnectionBroker')

    # Get the current state using our updated Get-TargetResource (which has the 2025 fixes)
    $currentState = Get-TargetResource -CollectionName $CollectionName -ConnectionBroker $cb

    # Standard DSC comparison logic
    $testDscParameterStateSplat = @{
        CurrentValues       = $currentState
        DesiredValues       = $PSBoundParameters
        TurnOffTypeChecking = $true
        SortArrayValues     = $true
        Verbose             = $VerbosePreference
    }

    return Test-DscParameterState @testDscParameterStateSplat
}
Export-ModuleMember -Function *-TargetResource