$modulePath = Join-Path -Path (Split-Path -Path (Split-Path -Path $PSScriptRoot -Parent) -Parent) -ChildPath 'Modules'

# Import the Common Modules
Import-Module -Name (Join-Path -Path $modulePath -ChildPath 'RemoteDesktopServicesDsc.Common')
Import-Module -Name (Join-Path -Path $modulePath -ChildPath 'DscResource.Common')

# Force the process to refresh the system module path immediately
$env:PSModulePath = [Environment]::GetEnvironmentVariable("PSModulePath", "Machine") + ";" + $env:PSModulePath

if (-not (Test-RemoteDesktopServicesDscOsRequirement))
{
    throw 'The minimum OS requirement was not met.'
}

# Helper function to ensure module visibility in Server 2025 restricted sessions
function Sync-RDSModule {
    $manifest = Join-Path $env:SystemRoot "system32\WindowsPowerShell\v1.0\Modules\RemoteDesktop\RemoteDesktop.psd1"
    if (Test-Path $manifest) {
        Import-Module -Name $manifest -ErrorAction SilentlyContinue
    }
}

#######################################################################
# The Get-TargetResource cmdlet.
#######################################################################
function Get-TargetResource
{
    [CmdletBinding()]
    [OutputType([System.Collections.Hashtable])]
    param
    (
        [Parameter(Mandatory = $true)]
        [System.String[]] $SessionHost,

        [Parameter(Mandatory = $true)]
        [System.String] $ConnectionBroker,

        [Parameter(Mandatory = $true)]
        [System.String[]] $WebAccessServer
    )

    Write-Verbose 'Getting list of RD Server roles.'

    # Direct import instead of Assert-Module
    Sync-RDSModule

    # Start service RDMS with wait logic for Server 2025 WID initialization
    $rdms = Get-Service -Name RDMS -ErrorAction SilentlyContinue
    if ($null -ne $rdms -and $rdms.Status -ne 'Running')
    {
        try {
            Start-Service -Name RDMS -ErrorAction Stop
            # Wait up to 20 seconds for the service to actually reach 'Running'
            $rdms.WaitForStatus('Running', '00:00:20')
            Start-Sleep -Seconds 5 # Buffer for database mounting
        }
        catch {
            Write-Warning "Failed to start RDMS service. Error: '$_'."
        }
    }

    # SERVER 2025 FIX: Use Fully Qualified Name
    $deployed = RemoteDesktop\Get-RDServer -ConnectionBroker $ConnectionBroker -ErrorAction SilentlyContinue

    @{
        SessionHost      = [System.String[]] ($deployed | Where-Object Roles -Contains 'RDS-RD-SERVER' | ForEach-Object Server)
        ConnectionBroker = $deployed | Where-Object Roles -Contains 'RDS-CONNECTION-BROKER' | ForEach-Object Server
        WebAccessServer  = [System.String[]] ($deployed | Where-Object Roles -Contains 'RDS-WEB-ACCESS' | ForEach-Object Server)
    }
}

########################################################################
# The Set-TargetResource cmdlet.
########################################################################
function Set-TargetResource
{
    [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidGlobalVars', 'global:DSCMachineStatus')]
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [System.String[]] $SessionHost,

        [Parameter(Mandatory = $true)]
        [System.String] $ConnectionBroker,

        [Parameter(Mandatory = $true)]
        [System.String[]] $WebAccessServer
    )

    Sync-RDSModule

    $currentStatus = Get-TargetResource @PSBoundParameters

    if ($null -eq $currentStatus.ConnectionBroker)
    {
        Write-Verbose 'Initiating new RDSH deployment.'
        $parameters = @{
            ConnectionBroker = $ConnectionBroker
            SessionHost      = $SessionHost
            WebAccessServer  = $WebAccessServer | Select-Object -First 1
        }

        # SERVER 2025 FIX: Use Fully Qualified Name
        RemoteDesktop\New-RDSessionDeployment @parameters
        $global:DSCMachineStatus = 1
        return
    }

    foreach ($server in ($SessionHost | Where-Object { $_ -notin $currentStatus.SessionHost }))
    {
        Write-Verbose "Adding server '$server' to deployment."
        RemoteDesktop\Add-RDServer -Server $server -Role 'RDS-RD-SERVER' -ConnectionBroker $ConnectionBroker
    }

    foreach ($server in ($currentStatus.SessionHost | Where-Object { $_ -notin $SessionHost }))
    {
        Write-Verbose "Removing server '$server' from deployment."
        RemoteDesktop\Remove-RDServer -Server $server -Role 'RDS-RD-SERVER' -ConnectionBroker $ConnectionBroker -Force
    }

    foreach ($server in ($WebAccessServer | Select-Object -Skip 1 | Where-Object { $_ -notin $currentStatus.WebAccessServer }))
    {
        Write-Verbose "Adding server '$server' to deployment."
        RemoteDesktop\Add-RDServer -Server $server -Role 'RDS-WEB-ACCESS' -ConnectionBroker $ConnectionBroker
    }

    foreach ($server in ($currentStatus.WebAccessServer | Where-Object { $_ -notin $WebAccessServer }))
    {
        Write-Verbose "Removing Web Server '$server' from deployment."
        RemoteDesktop\Remove-RDServer -Server $server -Role 'RDS-WEB-ACCESS' -ConnectionBroker $ConnectionBroker -Force
    }
}

#######################################################################
# The Test-TargetResource cmdlet.
#######################################################################
function Test-TargetResource
{
    [CmdletBinding()]
    [OutputType([System.Boolean])]
    param
    (
        [Parameter(Mandatory = $true)]
        [System.String[]] $SessionHost,

        [Parameter(Mandatory = $true)]
        [System.String] $ConnectionBroker,

        [Parameter(Mandatory = $true)]
        [System.String[]] $WebAccessServer
    )

    Write-Verbose 'Checking RDSH role is deployed on this node.'

    $desiredState = $PSBoundParameters
    $currentState = Get-TargetResource @PSBoundParameters

    return Test-DscParameterState `
        -CurrentValues $currentState `
        -DesiredValues $desiredState `
        -SortArrayValues `
        -Verbose:$VerbosePreference
}

Export-ModuleMember -Function *-TargetResource